import sys

import csv
import shutil

from collections import defaultdict
import numpy



class Samis:
  """Summons a demon to copy data from Moodle spradsheets to samis spreadsheets. Remember to wear a type 1 hazmat suit whilst doing this."""
  
  def __init__(self):
    self._output = {} # name -> fn
    self._input = {} # fn -> (name, percentage, human friendly name)
    self._remap = {} # original STU code -> new STU code
    self._skip = set() # STU codes, as integers.
  
  
  def add_pentagram(self, name, fn):
    """Adds the samis spreadsheet with the given filename as an output. The name is used to match each mark spreadsheet to the assesment it belongs to."""
    self._output[name] = fn
  
  
  def add_goat(self, name, percentage, fn, human = None):
    self._input[fn] = (name, percentage, human if human is not None else fn)

  
  def remap(self, old, new):
    """For if Moodle contains STU codes that don't match the Samis ones. Yes, this is a common occurance."""
    self._remap[old] = new


  def skip(self, stu):
    """Given a STU code, without any of that silly slash nonsense, this ensures that student remains blank. For when they have been a bad student."""
    self._skip.add(int(stu))


  def sacrifice(self):
    """Performs the rite of transferance."""
    
    # Sum the percentages and print out some sanity info...
    total = defaultdict(float) # name -> percentage
    for name, percentage, human in self._input.values():
      total[name] += percentage
    
    print('Totals:')
    for name, percentage in sorted(total.items()):
      print('  {}: {:g}%'.format(name, percentage))
      if name not in self._output:
        print('  NO MATCHING OUTPUT')
        sys.exit(1)
    totals_sum = sum(total.values())
    print('Total = {:g}%'.format(totals_sum))
    print()
    
    
    # Data structure - dictionary indexed by STU (int), going to a dictionary of {name: total percentage}...
    data = defaultdict(lambda: defaultdict(float))
    
    
    # Loop all moodle spreeadsheets and read in the data from each...
    stu_to_name = {}
    for fn, (name, percentage, human) in sorted(self._input.items()):
      print('{}:'.format(human))
      
      samples = []
      
      with open(fn, newline='') as fin:
        first = True
        
        
        for row in csv.reader(fin):
          if first:
            index_stu = row.index('STU Code')
            index_grade = row.index('Grade')
            index_max = row.index('Maximum Grade')
            index_name = row.index('Full name')
            
            first = False
            continue
          
          if len(row[index_grade].strip()) > 0:
            stu = int(row[index_stu])
            perage = float(row[index_grade]) / float(row[index_max])
          
            data[stu][name] += percentage * perage
            
            if stu not in stu_to_name:
              stu_to_name[stu] = row[index_name]
            
            samples.append(perage)
      
      samples = numpy.array(samples)
      print('  students = {}, mean = {:g}%, std = {:g}% (worth {}%)'.format(samples.shape[0], samples.mean() * 100, samples.std() * 100, percentage))
      print()
    
    
    # Remap...
    for old, new in self._remap.items():
      if old in data:
        if new in data:
          print('Warning: Remap overwriting {}'.format(new))
        
        data[new] = data[old]
        del data[old]
    
    
    # Print out final marks summary statistics...
    marks = []
    for stu, names in data.items():
      mark = 0.0
      for name, percentage in names.items():
        mark += percentage
      
      mark = 100 * mark / totals_sum
      marks.append(mark)
    
    marks = numpy.array(marks)
    print('Final:')
    print('  students     = {}'.format(len(marks)))
    print('  failed (<40) = {}'.format(sum(marks<40)))
    print('  mean         = {:g}%'.format(marks.mean()))
    print('  std          = {:g}%'.format(marks.std()))
    print()
    
    
    # Backup samis spreadsheets...
    for fn in self._output.values():
      shutil.copyfile(fn, fn + '~')
    
    
    # Loop samis spreadsheets, reading and then rewriting each...
    used = set()
    for name, fn in self._output.items():
      # Read...
      original = []
      with open(fn, 'r', newline='') as fin:
        first = True
        
        for row in csv.reader(fin):
          if first:
            first = False
            
            index_stu = row.index('#Cand Key')
            index_mark = row.index('Mark')
            index_grade = row.index('Grade')
          
          original.append(row)
      
      # Write...
      with open(fn, 'w', newline='') as fout:
        writer = csv.writer(fout)
        first = True
        
        for row in original:
          if not first:
            stu = int(row[index_stu][1:].split('/')[0])
          
            if stu in data and stu not in self._skip and name in data[stu] and len(row[index_grade].strip())==0:
              row = list(row)
              row[index_mark] = '{:g}'.format(100 * data[stu][name] / total[name])
              used.add(stu)
              
          else:
            first = False
          writer.writerow(row)
          
    print('Goats sacrificed, demon summoned. All hail the dark lord Samis!')
    print()
    
    if len(used)!=len(data):
      print('Spare chickens:')
      for stu in data:
        if stu not in used and stu not in self._skip:
          print('Chicken {} -- {}'.format(stu, format(stu_to_name.get(stu, 'Unknown'))))
