#! /usr/bin/env python3

import re
from collections import defaultdict

import numpy
from scipy.stats import chisquare

import automark2 as am



# Need these later...
cake = {'Banoffee Pie' : (0, 1 / 3.8), 'Butterfly Cake' : (1, 1 / 1), 'Caterpillar Cake' : (2, 1 / 20), 'Lava Cake' : (3, 1 / 4.2), 'Tiramisu' : (4, 1 / 3.5)}
coffee = {'Espresso' : (0, 2.0), 'Flat White' : (1, 2.8), 'Americano' : (2, 2.4), 'Latte' : (3, 3.2)}



# A reaction function that modifies 'go_buy' in the Customer object to record all bought items into the timeline object indepently of the students code...
def alt_go_buy(self, timeline, buy):
  # Original code...
  self.buy = buy 
  timeline.join_queue('tills', self)
  
  # Record the buy list in a weird timeline variable...
  if not hasattr(timeline, 'marking_bought'):
    timeline.marking_bought = defaultdict(int)
  
  for item in buy:
    timeline.marking_bought[item] += 1


def modify_go_buy(state):
  if 'Customer' in state:
    state['Customer'].go_buy = alt_go_buy



# Load code...
notebook = am.Notebook(reactions = [modify_go_buy])



# Q1 - two cashiers...

## Function that checks for the existance of a timeline with two cashiers in a cell...
Timeline = notebook.state()['Timeline']
Cashier = notebook.state()['Cashier']

def dual_cashiers(cell):
  for value in cell.state().values():
    if isinstance(value, Timeline):
      cashiers = 0
      for closure in value.queue['tills']['sink']:
        if isinstance(closure.__self__, Cashier):
          cashiers += 1

      if cashiers >= 2:
        return True
  
  return False


## Question...
q1 = am.Question(1, 2)

q1.worth(None, 2)
q1.add(None, dual_cashiers)

q1(notebook)



# Q2 - purchase statistics...

## Code that checks cells for having printed out counts and a total that match the values captured by the above modification...
def prints_sold(cell, inc_coffee = False):
  numbers = [float(v) for v in re.findall(r'\d+(?:\.\d*)?', cell.output())]
  
  for value in cell.state().values():
    if isinstance(value, Timeline) and hasattr(value, 'marking_bought'):
      if inc_coffee and not numpy.any([key in coffee for key in value.marking_bought]):
        continue
      
      match = True
      menu = cell.state()['menu']
      
      total = 0.0
      for key, value in value.marking_bought.items():
        if not numpy.any(numpy.isclose(value, numbers)):
          match = False
          break
        
        total += menu[key][0] * value
      
      if not numpy.any(numpy.isclose(total, numbers)):
        match = False
      
      if match:
        return True
  
  return None


## Turn above into a question, noting that it only gives all marks or indicates uncertainty, as it can't do partial marks...
q2 = am.Question(2, 4)

q2.worth(None, 4)
q2.add(None, prints_sold)

q2(notebook)



# Q3 - coffee shop...
# (Can't really check this one beyond that it works 100% - most of below is really hints except for last where it checks that customers got coffee!)

## For detecting that a barista exists in a timeline...
try:
  Barista = notebook.state()['Barista']
  
except KeyError:
  class Barista:
    pass


def has_barista(cell):
  for value in cell.state().values():
    if isinstance(value, Timeline):
      for closure in value.queue['brew']['sink']:
        if isinstance(closure.__self__, Barista):
          return True
  
  return False


## For detecting all customers exited...
def all_served(cell):
  for value in cell.state().values():
    if isinstance(value, Timeline):
      arrive = len(value.events['arrive'])
      leave = len(value.events['leave'])
      if arrive>0 and arrive==leave:
        return True
  
  return False


## The actual question...
q3 = am.Question(3, 15)

q3.worth('ServeCustomer', 2)
q3.add('ServeCustomer', am.CodeMatch(['class ServeCustomer', 'def __init__', 'join_queue(name = "brew", obj)']))

q3.worth('Brew', 8)
q3.add('Brew', am.CodeMatch(['class Brew']))
q3.add('Brew', am.CodeMatch(['class Brew', 'def __init__']))
q3.add('Brew', am.CodeMatch(['class Brew', 'def time']))
q3.add('Brew', am.CodeMatch(['class Brew', 'def depends']))
q3.add('Brew', am.CodeMatch(['class Brew', 'def __call__']))
q3.add('Brew', am.CodeMatch(['class Brew', 'def __call__', 'receive_item()']))

q3.worth('Barista', 4)
q3.add('Barista', am.CodeMatch(['class Barista']))

q3.worth('Running', 1)
q3.add('Running', am.Uncertain(am.All(has_barista,
                                      all_served)))

q3(notebook)



# Q4 - everyone wants coffee...
ChoosePurchase = notebook.state()['ChoosePurchase']

## Collect lots of sample purchases...
purchases = []

class FakeCustomer:
  def go_buy(self, timeline, buy):
    global purchases
    purchases.append(buy)

for _ in range(1024 * 8):
  fc = FakeCustomer()
  cp = ChoosePurchase(fc)
  cp(None)
  del cp

## Some tests...
def group_prob(injest):
  # Count number of coffees in each order and build histogram - must be in [1, 4] so fail if not...
  counts = numpy.zeros(4) # Offset by 1 for indices
  for buy in purchases:
    count = sum(item in coffee for item in buy)
    if count < 1 or count>4:
      print('Warning: Invalid coffee count in order')
      return None
    counts[count - 1] += 1
  
  # Do Chi-squared and verify that it matches - if not then return None...
  expect = counts.sum() * numpy.array([0.7, 0.1, 0.1, 0.1])
  _, pval = chisquare(counts, expect)
  
  if pval>0.05:
    return True
 
  else:
    return None


def cake_prob(injest):
  # Count how many times the groups get cake, checking it's all or none...
  all_cake = 0
  for buy in purchases:
    cake_count = sum(item in cake for item in buy)
    coffee_count = sum(item in coffee for item in buy)
    
    if cake_count!=0:
      if cake_count!=coffee_count:
        print('Warning: Non-zero cake count does not match group size')
        return None
      all_cake += 1
  
  counts = [len(purchases) - all_cake, all_cake]
  _, pval = chisquare(counts)
  
  if pval>0.05:
    return True
 
  else:
    return None



def cake_dist(injest):
  # Count the kinds of cake...
  counts = numpy.zeros(len(cake))
  for buy in purchases:
    for item in buy:
      if item in cake:
        counts[cake[item][0]] += 1
  
  # Do Chi-squared and verify that it matches - if not then return None...
  expect = numpy.zeros(len(cake))
  for index, prob in cake.values():
    expect[index] = prob
  
  expect /= expect.sum()
  expect *= counts.sum()
  
  _, pval = chisquare(counts, expect)
  
  if pval>0.05:
    return True
 
  else:
    return None


def coffee_dist(injest):
  # Count the kinds of coffee...
  counts = numpy.zeros(len(coffee))
  for buy in purchases:
    for item in buy:
      if item in coffee:
        counts[coffee[item][0]] += 1
  
  # Do Chi-squared and verify that it matches - if not then return None...
  expect = numpy.zeros(len(coffee))
  for index, prob in coffee.values():
    expect[index] = prob
  
  expect /= expect.sum()
  expect *= counts.sum()
  
  _, pval = chisquare(counts, expect)
  
  if pval>0.05:
    return True
 
  else:
    return None


## Actual question...
q4 = am.Question(4, 5)

q4.worth('GroupCake', 2)
q4.add('GroupCake', group_prob, 'end')
q4.add('GroupCake', cake_prob, 'end')

q4.worth('PropDist', 2)
q4.add('PropDist', cake_dist, 'end')
q4.add('PropDist', coffee_dist, 'end')

q4.worth('Graph', 1)
q4.add('Graph', lambda cell: prints_sold(cell, True))

q4(notebook)
